<?php
/**
 * Plugin Name: FX Copy Host (Simple)
 * Plugin URI: https://example.com/
 * Description: Simple REST API host for MT4 copy trading (1 master, 1 slave).
 * Version: 1.0.0
 * Author: ChatGPT & HIRO
 * License: GPL2
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('FXCopyHost')) {

class FXCopyHost {

    const OPTION_API_KEY        = 'fxcopy_host_api_key';
    const OPTION_MASTER_ACCOUNT = 'fxcopy_host_master_account';
    const TABLE_NAME            = 'fxcopy_signals';

    /**
     * プラグイン有効化時：テーブル作成＆APIキー自動生成
     */
    public static function activate() {
        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $charset_collate = $wpdb->get_charset_collate();

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        $sql = "CREATE TABLE $table_name (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            master_account VARCHAR(64) NOT NULL,
            api_key VARCHAR(64) NOT NULL,
            signal_type VARCHAR(16) NOT NULL,
            ticket BIGINT(20) UNSIGNED NOT NULL,
            symbol VARCHAR(32) NOT NULL,
            cmd TINYINT NOT NULL,
            lots DOUBLE NOT NULL,
            price DOUBLE NOT NULL,
            sl DOUBLE NOT NULL,
            tp DOUBLE NOT NULL,
            magic BIGINT(20) NOT NULL,
            comment VARCHAR(191) NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            processed TINYINT(1) NOT NULL DEFAULT 0,
            PRIMARY KEY  (id),
            KEY master_account (master_account),
            KEY processed (processed)
        ) $charset_collate;";

        dbDelta($sql);

        if (!get_option(self::OPTION_API_KEY)) {
            if (function_exists('wp_generate_password')) {
                $api_key = wp_generate_password(24, false, false);
            } else {
                $api_key = substr(md5(uniqid('', true)), 0, 24);
            }
            add_option(self::OPTION_API_KEY, $api_key);
        }

        if (!get_option(self::OPTION_MASTER_ACCOUNT)) {
            add_option(self::OPTION_MASTER_ACCOUNT, 'MASTER001');
        }
    }

    /**
     * 管理画面メニュー追加
     */
    public static function add_admin_menu() {
        add_options_page(
            'FX Copy Host',
            'FX Copy Host',
            'manage_options',
            'fxcopy-host',
            array(__CLASS__, 'render_settings_page')
        );
    }

    /**
     * 設定ページ描画
     */
    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_POST['fxcopy_host_save']) && check_admin_referer('fxcopy_host_save_settings')) {
            $api_key        = isset($_POST['fxcopy_api_key']) ? sanitize_text_field($_POST['fxcopy_api_key']) : '';
            $master_account = isset($_POST['fxcopy_master_account']) ? sanitize_text_field($_POST['fxcopy_master_account']) : '';

            if ($api_key !== '') {
                update_option(self::OPTION_API_KEY, $api_key);
            }
            if ($master_account !== '') {
                update_option(self::OPTION_MASTER_ACCOUNT, $master_account);
            }

            echo '<div class="updated"><p>設定を保存しました。</p></div>';
        }

        $api_key        = esc_attr(get_option(self::OPTION_API_KEY));
        $master_account = esc_attr(get_option(self::OPTION_MASTER_ACCOUNT));
        ?>
        <div class="wrap">
            <h1>FX Copy Host 設定</h1>
            <form method="post">
                <?php wp_nonce_field('fxcopy_host_save_settings'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="fxcopy_api_key">APIキー</label></th>
                        <td>
                            <input type="text" id="fxcopy_api_key" name="fxcopy_api_key" value="<?php echo $api_key; ?>" size="40" />
                            <p class="description">MT4 EA側の ApiKey と同じ値を設定してください。</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="fxcopy_master_account">マスターアカウントID</label></th>
                        <td>
                            <input type="text" id="fxcopy_master_account" name="fxcopy_master_account" value="<?php echo $master_account; ?>" size="40" />
                            <p class="description">MT4 EA側の MasterAccount と同じ値を設定してください。</p>
                        </td>
                    </tr>
                </table>
                <p class="submit">
                    <button type="submit" name="fxcopy_host_save" class="button button-primary">保存</button>
                </p>
            </form>

            <h2>接続情報（MT4 EA用）</h2>
            <p>Push URL（マスターEA → サーバー）:<br>
               <code><?php echo esc_html( rest_url('fxcopy/v1/push') ); ?></code></p>
            <p>Pull URL（スレーブEA → サーバー）:<br>
               <code><?php echo esc_html( rest_url('fxcopy/v1/pull') ); ?></code></p>
        </div>
        <?php
    }

    /**
     * REST APIルート登録
     */
    public static function register_routes() {
        register_rest_route(
            'fxcopy/v1',
            '/push',
            array(
                'methods'             => 'POST',
                'callback'            => array(__CLASS__, 'handle_push'),
                'permission_callback' => '__return_true',
            )
        );

        register_rest_route(
            'fxcopy/v1',
            '/pull',
            array(
                'methods'             => 'GET',
                'callback'            => array(__CLASS__, 'handle_pull'),
                'permission_callback' => '__return_true',
            )
        );
    }

    /**
     * APIキーとマスターIDのチェック
     */
    protected static function validate_key_and_master($api_key, $master_account) {
        $opt_key    = get_option(self::OPTION_API_KEY);
        $opt_master = get_option(self::OPTION_MASTER_ACCOUNT);

        if (empty($opt_key) || empty($opt_master)) {
            return false;
        }
        if ($api_key !== $opt_key) {
            return false;
        }
        if ($master_account !== $opt_master) {
            return false;
        }
        return true;
    }

    /**
     * /push エンドポイント：マスターEAからのシグナル受信
     */
    public static function handle_push($request) {
        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_NAME;

        $api_key        = $request->get_param('api_key');
        $master_account = $request->get_param('master_account');

        if (!self::validate_key_and_master($api_key, $master_account)) {
            return new WP_REST_Response(array('error' => 'invalid api_key or master_account'), 403);
        }

        $data = array(
            'master_account' => $master_account,
            'api_key'        => $api_key,
            'signal_type'    => sanitize_text_field($request->get_param('signal_type')),
            'ticket'         => intval($request->get_param('ticket')),
            'symbol'         => sanitize_text_field($request->get_param('symbol')),
            'cmd'            => intval($request->get_param('cmd')),
            'lots'           => floatval($request->get_param('lots')),
            'price'          => floatval($request->get_param('price')),
            'sl'             => floatval($request->get_param('sl')),
            'tp'             => floatval($request->get_param('tp')),
            'magic'          => intval($request->get_param('magic')),
            'comment'        => sanitize_text_field($request->get_param('comment')),
            'created_at'     => current_time('mysql'),
            'processed'      => 0,
        );

        $formats = array(
            '%s', // master_account
            '%s', // api_key
            '%s', // signal_type
            '%d', // ticket
            '%s', // symbol
            '%d', // cmd
            '%f', // lots
            '%f', // price
            '%f', // sl
            '%f', // tp
            '%d', // magic
            '%s', // comment
            '%s', // created_at
            '%d', // processed
        );

        $result = $wpdb->insert($table_name, $data, $formats);

        if ($result === false) {
            return new WP_REST_Response(array('error' => 'insert failed'), 500);
        }

        return array('result' => 'ok');
    }

    /**
     * /pull エンドポイント：スレーブEAからのシグナル取得
     */
    public static function handle_pull($request) {
        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_NAME;

        $api_key        = $request->get_param('api_key');
        $master_account = $request->get_param('master_account');

        if (!self::validate_key_and_master($api_key, $master_account)) {
            return new WP_REST_Response(array('error' => 'invalid api_key or master_account'), 403);
        }

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, signal_type, ticket, symbol, cmd, lots, price, sl, tp, magic, comment
                 FROM $table_name
                 WHERE master_account = %s AND api_key = %s AND processed = 0
                 ORDER BY id ASC
                 LIMIT 50",
                $master_account,
                $api_key
            ),
            ARRAY_A
        );

        if (!empty($rows)) {
            $ids = array();
            foreach ($rows as $row) {
                $ids[] = intval($row['id']);
            }
            if (!empty($ids)) {
                $id_list = implode(',', $ids);
                $wpdb->query("UPDATE $table_name SET processed = 1 WHERE id IN ($id_list)");
            }
        }

        return array('signals' => $rows);
    }
}

} // end class exists

register_activation_hook(__FILE__, array('FXCopyHost', 'activate'));
add_action('admin_menu', array('FXCopyHost', 'add_admin_menu'));
add_action('rest_api_init', array('FXCopyHost', 'register_routes'));
