//+------------------------------------------------------------------+
//|                                              MT4Monitor_Logger.mq4 |
//| MT4 -> WordPress : ping ＆ trade log                               |
//+------------------------------------------------------------------+
#property strict

input string InpApiBaseUrl      = "https://www.example.com/wp-json/mt4monitor/v1";
input string InpApiKey          = "CHANGE_ME";
input int    InpPingIntervalSec = 60;   // ping間隔（秒）
input bool   InpSendTrades      = true; // クローズ注文送信
input int    InpTimeoutMs       = 5000; // WebRequestタイムアウト(ms)

datetime g_last_ping_time      = 0;
datetime g_last_trade_synctime = 0;

//+------------------------------------------------------------------+
int OnInit()
  {
   g_last_ping_time      = 0;
   g_last_trade_synctime = TimeCurrent(); // EA起動以降のクローズ注文を対象
   EventSetTimer(10); // 10秒ごとにOnTimer
   Print("MT4Monitor_Logger initialized.");
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   EventKillTimer();
   Print("MT4Monitor_Logger deinitialized.");
  }
//+------------------------------------------------------------------+
void OnTimer()
  {
   datetime now = TimeCurrent();

   if(now - g_last_ping_time >= InpPingIntervalSec)
     {
      SendHeartbeat();
      g_last_ping_time = now;

      if(InpSendTrades)
         SyncClosedTrades();
     }
  }
//+------------------------------------------------------------------+
// フォームPOST (application/x-www-form-urlencoded)
//+------------------------------------------------------------------+
int HttpPostForm(string url, string body, string &response)
  {
   string headers = "Content-Type: application/x-www-form-urlencoded\r\n";

   char post_data[];
   char result_data[];
   string result_headers = "";

   int len = StringLen(body);
   ArrayResize(post_data, len);
   StringToCharArray(body, post_data, 0, len);

   ResetLastError();
   int status = WebRequest("POST", url, headers, InpTimeoutMs,
                           post_data, result_data, result_headers);
   int err = GetLastError();

   response = CharArrayToString(result_data, 0, ArraySize(result_data));

   if(status == -1)
     {
      Print("WebRequest error. err=", err, " url=", url);
      return(-1);
     }

   if(status != 200)
     {
      Print("HTTP error. status=", status, " resp=", response);
     }

   return(status);
  }
//+------------------------------------------------------------------+
// ping送信
//+------------------------------------------------------------------+
void SendHeartbeat()
  {
   string url = InpApiBaseUrl + "/ping";

   double balance = AccountBalance();
   double equity  = AccountEquity();
   double margin  = AccountMargin();
   double free    = AccountFreeMargin();

   string body = StringFormat(
      "api_key=%s&account_id=%d&server=%s&platform=mt4&balance=%s&equity=%s&margin=%s&margin_free=%s",
      InpApiKey,
      AccountNumber(),
      AccountServer(),
      DoubleToString(balance,2),
      DoubleToString(equity,2),
      DoubleToString(margin,2),
      DoubleToString(free,2)
   );

   string resp;
   int status = HttpPostForm(url, body, resp);
   if(status == 200)
     {
      //Print("Heartbeat OK: ", resp);
     }
   else
     {
      Print("Heartbeat failed. status=", status, " resp=", resp);
     }
  }
//+------------------------------------------------------------------+
// クローズ注文をWordPressへ送信（このEA起動以降）
//+------------------------------------------------------------------+
void SyncClosedTrades()
  {
   datetime newest_close = g_last_trade_synctime;
   int total = OrdersHistoryTotal();

   for(int i = total - 1; i >= 0; i--)
     {
      if(!OrderSelect(i, SELECT_BY_POS, MODE_HISTORY))
         continue;

      datetime ctime = OrderCloseTime();
      if(ctime <= g_last_trade_synctime)
         continue;

      SendClosedTrade(OrderTicket(), ctime);

      if(ctime > newest_close)
         newest_close = ctime;
     }

   if(newest_close > g_last_trade_synctime)
      g_last_trade_synctime = newest_close;
  }
//+------------------------------------------------------------------+
// 1件のクローズ注文を送信
//+------------------------------------------------------------------+
void SendClosedTrade(int ticket, datetime close_time)
  {
   if(!OrderSelect(ticket, SELECT_BY_TICKET, MODE_HISTORY))
     {
      Print("SendClosedTrade: OrderSelect failed ticket=", ticket);
      return;
     }

   string url = InpApiBaseUrl + "/trade";

   int    acc       = AccountNumber();
   string symbol    = OrderSymbol();
   int    type      = OrderType();
   double lots      = OrderLots();
   datetime ot      = OrderOpenTime();
   double  op       = OrderOpenPrice();
   double  cp       = OrderClosePrice();
   double  profit   = OrderProfit() + OrderSwap() + OrderCommission();
   int     magic    = OrderMagicNumber();

   string open_time_str  = TimeToStr(ot, TIME_DATE|TIME_SECONDS);
   string close_time_str = TimeToStr(close_time, TIME_DATE|TIME_SECONDS);

   string body = StringFormat(
      "api_key=%s&account_id=%d&ticket=%d&symbol=%s&type=%d&lots=%s"
      "&open_time=%s&close_time=%s"
      "&open_price=%s&close_price=%s"
      "&profit=%s&magic=%d",
      InpApiKey,
      acc,
      ticket,
      symbol,
      type,
      DoubleToString(lots,2),
      open_time_str,
      close_time_str,
      DoubleToString(op,Digits),
      DoubleToString(cp,Digits),
      DoubleToString(profit,2),
      magic
   );

   string resp;
   int status = HttpPostForm(url, body, resp);
   if(status == 200)
     {
      //Print("Trade sent. ticket=", ticket);
     }
   else
     {
      Print("Trade send failed. ticket=", ticket, " status=", status, " resp=", resp);
     }
  }
//+------------------------------------------------------------------+
