<?php
/*
Plugin Name: MT4 Monitor & Trade Logger
Description: Receive heartbeat pings and trade logs from MT4, notify downtime, store trade history, and display results.
Version: 0.3.0
Author: HIRO + ChatGPT
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'MT4MON_OPTION_API_KEY',   'mt4mon_api_key' );
define( 'MT4MON_OPTION_THRESHOLD', 'mt4mon_down_threshold' );
define( 'MT4MON_OPTION_EMAIL',     'mt4mon_notify_email' );

function mt4mon_table_accounts() {
    global $wpdb;
    return $wpdb->prefix . 'mt4mon_accounts';
}

function mt4mon_table_trades() {
    global $wpdb;
    return $wpdb->prefix . 'mt4mon_trades';
}

/**
 * 5分間隔スケジュール追加
 */
add_filter( 'cron_schedules', function( $schedules ) {
    if ( ! isset( $schedules['mt4mon_five_minutes'] ) ) {
        $schedules['mt4mon_five_minutes'] = array(
            'interval' => 300,
            'display'  => 'Every 5 Minutes (MT4 Monitor)',
        );
    }
    return $schedules;
});

/**
 * 有効化時：テーブル作成＋Cron登録
 */
function mt4mon_activate() {
    global $wpdb;
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';

    $charset_collate = $wpdb->get_charset_collate();

    $sql_accounts = "CREATE TABLE " . mt4mon_table_accounts() . " (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        account_id bigint(20) NOT NULL,
        server varchar(191) NOT NULL,
        platform varchar(10) NOT NULL DEFAULT 'mt4',
        last_ping datetime NULL,
        last_ip varchar(45) NULL,
        status varchar(20) NOT NULL DEFAULT 'up',
        last_status_change datetime NULL,
        created_at datetime NOT NULL,
        updated_at datetime NOT NULL,
        PRIMARY KEY  (id),
        UNIQUE KEY account_unique (account_id, server)
    ) $charset_collate;";

    $sql_trades = "CREATE TABLE " . mt4mon_table_trades() . " (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        account_id bigint(20) NOT NULL,
        ticket bigint(20) NOT NULL,
        symbol varchar(32) NOT NULL,
        type tinyint(3) NOT NULL,
        lots double NOT NULL,
        open_time datetime NOT NULL,
        close_time datetime NOT NULL,
        open_price double NOT NULL,
        close_price double NOT NULL,
        profit double NOT NULL,
        magic bigint(20) NOT NULL,
        comment varchar(191) DEFAULT '',
        created_at datetime NOT NULL,
        PRIMARY KEY (id),
        UNIQUE KEY trade_unique (account_id, ticket)
    ) $charset_collate;";

    dbDelta( $sql_accounts );
    dbDelta( $sql_trades );

    if ( ! wp_next_scheduled( 'mt4mon_cron_check_down' ) ) {
        wp_schedule_event( time(), 'mt4mon_five_minutes', 'mt4mon_cron_check_down' );
    }

    if ( ! get_option( MT4MON_OPTION_THRESHOLD ) ) {
        update_option( MT4MON_OPTION_THRESHOLD, 10 );
    }
    if ( ! get_option( MT4MON_OPTION_EMAIL ) ) {
        update_option( MT4MON_OPTION_EMAIL, get_option( 'admin_email' ) );
    }
    if ( ! get_option( MT4MON_OPTION_API_KEY ) ) {
        update_option( MT4MON_OPTION_API_KEY, wp_generate_password( 24, false ) );
    }
}
register_activation_hook( __FILE__, 'mt4mon_activate' );

/**
 * 無効化時：Cron解除
 */
function mt4mon_deactivate() {
    wp_clear_scheduled_hook( 'mt4mon_cron_check_down' );
}
register_deactivation_hook( __FILE__, 'mt4mon_deactivate' );

/**
 * REST API ルート登録
 */
add_action( 'rest_api_init', function() {
    register_rest_route(
        'mt4monitor/v1',
        '/ping',
        array(
            'methods'             => 'POST',
            'callback'            => 'mt4mon_rest_ping',
            'permission_callback' => '__return_true',
        )
    );

    register_rest_route(
        'mt4monitor/v1',
        '/trade',
        array(
            'methods'             => 'POST',
            'callback'            => 'mt4mon_rest_trade',
            'permission_callback' => '__return_true',
        )
    );
});

/**
 * APIキー検証
 */
function mt4mon_check_api_key( $key ) {
    $expected = get_option( MT4MON_OPTION_API_KEY );
    if ( ! $expected ) {
        return false;
    }
    return hash_equals( (string) $expected, (string) $key );
}

/**
 * /ping 受信（ダウン→復旧時にメール通知）
 */
function mt4mon_rest_ping( WP_REST_Request $request ) {
    global $wpdb;

    $api_key    = (string) $request->get_param( 'api_key' );
    $account_id = intval( $request->get_param( 'account_id' ) );
    $server     = (string) $request->get_param( 'server' );
    $platform   = (string) $request->get_param( 'platform' );

    if ( $platform === '' ) {
        $platform = 'mt4';
    }

    if ( ! mt4mon_check_api_key( $api_key ) ) {
        return new WP_REST_Response(
            array( 'error' => 'invalid_api_key' ),
            403
        );
    }

    if ( ! $account_id || $server === '' ) {
        return new WP_REST_Response(
            array(
                'error' => 'missing_account',
                'debug' => array(
                    'account_id' => $account_id,
                    'server'     => $server,
                ),
            ),
            400
        );
    }

    $now_gmt = current_time( 'mysql', 1 );

    $ip = $request->get_header( 'x-forwarded-for' );
    if ( ! $ip ) {
        $ip = $request->get_header( 'remote_addr' );
    }
    if ( ! $ip && isset( $_SERVER['REMOTE_ADDR'] ) ) {
        $ip = $_SERVER['REMOTE_ADDR'];
    }

    $table = mt4mon_table_accounts();

    $existing = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT * FROM $table WHERE account_id = %d AND server = %s",
            $account_id,
            $server
        )
    );

    $was_down = false;

    if ( $existing ) {
        if ( isset( $existing->status ) && $existing->status === 'down' ) {
            $was_down = true;
        }

        $wpdb->update(
            $table,
            array(
                'platform'           => $platform,
                'last_ping'          => $now_gmt,
                'last_ip'            => $ip,
                'status'             => 'up',
                'updated_at'         => $now_gmt,
                'last_status_change' => ( $existing->status === 'up' ? $existing->last_status_change : $now_gmt ),
            ),
            array( 'id' => $existing->id ),
            array( '%s', '%s', '%s', '%s', '%s', '%s' ),
            array( '%d' )
        );
    } else {
        $wpdb->insert(
            $table,
            array(
                'account_id'         => $account_id,
                'server'             => $server,
                'platform'           => $platform,
                'last_ping'          => $now_gmt,
                'last_ip'            => $ip,
                'status'             => 'up',
                'last_status_change' => $now_gmt,
                'created_at'         => $now_gmt,
                'updated_at'         => $now_gmt,
            ),
            array( '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s' )
        );
    }

    // ダウン → 復旧時の通知
    if ( $was_down ) {
        $notify_email = get_option( MT4MON_OPTION_EMAIL, get_option( 'admin_email' ) );
        if ( $notify_email ) {
            $subject = sprintf( '[MT4Monitor] MT4 UP: %d (%s)', $account_id, $server );
            $body    = "MT4 からの ping が復旧しました。\n\n" .
                "Account: {$account_id}\n" .
                "Server : {$server}\n" .
                "Time   (GMT): {$now_gmt}\n\n" .
                "MT4 / VPS が正常に動作しているかを確認してください。";
            wp_mail( $notify_email, $subject, $body );
        }
    }

    return new WP_REST_Response(
        array(
            'result'     => 'ok',
            'account_id' => $account_id,
            'server'     => $server,
            'time'       => $now_gmt,
        ),
        200
    );
}

/**
 * /trade 受信
 */
function mt4mon_rest_trade( WP_REST_Request $request ) {
    global $wpdb;

    $api_key    = (string) $request->get_param( 'api_key' );
    $account_id = intval( $request->get_param( 'account_id' ) );

    if ( ! mt4mon_check_api_key( $api_key ) ) {
        return new WP_REST_Response( array( 'error' => 'invalid_api_key' ), 403 );
    }

    if ( ! $account_id ) {
        return new WP_REST_Response( array( 'error' => 'missing_account' ), 400 );
    }

    $ticket      = intval( $request->get_param( 'ticket' ) );
    $symbol      = (string) $request->get_param( 'symbol' );
    $type        = intval( $request->get_param( 'type' ) );
    $lots        = floatval( $request->get_param( 'lots' ) );
    $open_time   = (string) $request->get_param( 'open_time' );
    $close_time  = (string) $request->get_param( 'close_time' );
    $open_price  = floatval( $request->get_param( 'open_price' ) );
    $close_price = floatval( $request->get_param( 'close_price' ) );
    $profit      = floatval( $request->get_param( 'profit' ) );
    $magic       = intval( $request->get_param( 'magic' ) );
    $comment     = (string) $request->get_param( 'comment' );

    if ( ! $ticket || $symbol === '' || $open_time === '' || $close_time === '' ) {
        return new WP_REST_Response(
            array(
                'error' => 'missing_fields',
                'debug' => array(
                    'ticket'     => $ticket,
                    'symbol'     => $symbol,
                    'open_time'  => $open_time,
                    'close_time' => $close_time,
                ),
            ),
            400
        );
    }

    $table   = mt4mon_table_trades();
    $now_gmt = current_time( 'mysql', 1 );

    $existing = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT id FROM $table WHERE account_id = %d AND ticket = %d",
            $account_id,
            $ticket
        )
    );
    if ( $existing ) {
        return new WP_REST_Response( array( 'result' => 'exists' ), 200 );
    }

    $wpdb->insert(
        $table,
        array(
            'account_id'  => $account_id,
            'ticket'      => $ticket,
            'symbol'      => $symbol,
            'type'        => $type,
            'lots'        => $lots,
            'open_time'   => $open_time,
            'close_time'  => $close_time,
            'open_price'  => $open_price,
            'close_price' => $close_price,
            'profit'      => $profit,
            'magic'       => $magic,
            'comment'     => $comment,
            'created_at'  => $now_gmt,
        ),
        array(
            '%d', '%d', '%s', '%d', '%f',
            '%s', '%s', '%f', '%f', '%f',
            '%d', '%s', '%s',
        )
    );

    return new WP_REST_Response( array( 'result' => 'ok', 'ticket' => $ticket ), 200 );
}

/**
 * ダウン検知 Cron
 */
add_action( 'mt4mon_cron_check_down', function() {
    global $wpdb;
    $table = mt4mon_table_accounts();

    $threshold_min = intval( get_option( MT4MON_OPTION_THRESHOLD, 10 ) );
    if ( $threshold_min <= 0 ) {
        $threshold_min = 10;
    }

    $now_ts = current_time( 'timestamp', 1 );
    $cutoff = gmdate( 'Y-m-d H:i:s', $now_ts - $threshold_min * 60 );

    $accounts = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM $table WHERE last_ping IS NOT NULL AND last_ping < %s",
            $cutoff
        )
    );

    if ( empty( $accounts ) ) {
        return;
    }

    $notify_email = get_option( MT4MON_OPTION_EMAIL, get_option( 'admin_email' ) );

    foreach ( $accounts as $acc ) {
        if ( $acc->status === 'down' ) {
            continue;
        }

        $subject = sprintf( '[MT4Monitor] MT4 DOWN: %d (%s)', $acc->account_id, $acc->server );
        $body    = "MT4 からの ping が途絶えました。\n\n" .
            "Account: {$acc->account_id}\n" .
            "Server : {$acc->server}\n" .
            "Last ping (GMT): {$acc->last_ping}\n\n" .
            "しばらくしても復旧しない場合は、ターミナルやVPSを確認してください。";

        if ( $notify_email ) {
            wp_mail( $notify_email, $subject, $body );
        }

        $now_gmt = gmdate( 'Y-m-d H:i:s', $now_ts );

        $wpdb->update(
            $table,
            array(
                'status'             => 'down',
                'last_status_change' => $now_gmt,
                'updated_at'         => $now_gmt,
            ),
            array( 'id' => $acc->id ),
            array( '%s', '%s', '%s' ),
            array( '%d' )
        );
    }
});

/**
 * 管理画面メニュー
 */
add_action( 'admin_menu', function() {
    add_options_page(
        'MT4 Monitor',
        'MT4 Monitor',
        'manage_options',
        'mt4mon-settings',
        'mt4mon_render_settings_page'
    );

    add_submenu_page(
        'options-general.php',
        'MT4トレード履歴',
        'MT4トレード履歴',
        'manage_options',
        'mt4mon-trades',
        'mt4mon_render_trades_page'
    );
});

/**
 * 設定登録
 */
add_action( 'admin_init', function() {
    register_setting( 'mt4mon_settings', MT4MON_OPTION_API_KEY );
    register_setting( 'mt4mon_settings', MT4MON_OPTION_THRESHOLD );
    register_setting( 'mt4mon_settings', MT4MON_OPTION_EMAIL );
});

/**
 * 設定ページ（アカウント状況）
 */
function mt4mon_render_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    global $wpdb;
    $accounts = $wpdb->get_results( "SELECT * FROM " . mt4mon_table_accounts() . " ORDER BY updated_at DESC LIMIT 50" );

    ?>
    <div class="wrap">
        <h1>MT4 Monitor & Trade Logger 設定</h1>
        <form method="post" action="options.php">
            <?php settings_fields( 'mt4mon_settings' ); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="mt4mon_api_key">API Key</label></th>
                    <td>
                        <input type="text" id="mt4mon_api_key" name="mt4mon_api_key"
                               value="<?php echo esc_attr( get_option( MT4MON_OPTION_API_KEY ) ); ?>"
                               class="regular-text" />
                        <p class="description">MT4 EA から送る api_key。第三者に知られないよう注意してください。</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="mt4mon_down_threshold">ダウン判定 (分)</label></th>
                    <td>
                        <input type="number" id="mt4mon_down_threshold" name="mt4mon_down_threshold"
                               value="<?php echo esc_attr( get_option( MT4MON_OPTION_THRESHOLD, 10 ) ); ?>"
                               class="small-text" />
                        <p class="description">最後の ping から何分経過したら「ダウン」とみなすか。</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="mt4mon_notify_email">通知先メールアドレス</label></th>
                    <td>
                        <input type="email" id="mt4mon_notify_email" name="mt4mon_notify_email"
                               value="<?php echo esc_attr( get_option( MT4MON_OPTION_EMAIL, get_option( 'admin_email' ) ) ); ?>"
                               class="regular-text" />
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>

        <h2>最近のアカウント状況（最新50件）</h2>
        <table class="widefat striped">
            <thead>
                <tr>
                    <th>Account</th>
                    <th>Server</th>
                    <th>Status</th>
                    <th>Last Ping (GMT)</th>
                    <th>Last IP</th>
                    <th>Updated</th>
                </tr>
            </thead>
            <tbody>
            <?php if ( $accounts ) : ?>
                <?php foreach ( $accounts as $acc ) : ?>
                    <tr>
                        <td><?php echo esc_html( $acc->account_id ); ?></td>
                        <td><?php echo esc_html( $acc->server ); ?></td>
                        <td><?php echo esc_html( strtoupper( $acc->status ) ); ?></td>
                        <td><?php echo esc_html( $acc->last_ping ); ?></td>
                        <td><?php echo esc_html( $acc->last_ip ); ?></td>
                        <td><?php echo esc_html( $acc->updated_at ); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php else : ?>
                <tr><td colspan="6">まだデータがありません。</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

/**
 * 管理画面：トレード履歴ビューア
 */
function mt4mon_render_trades_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    global $wpdb;
    $table = mt4mon_table_trades();

    $account_id = isset( $_GET['account_id'] ) ? intval( $_GET['account_id'] ) : 0;
    $symbol     = isset( $_GET['symbol'] ) ? sanitize_text_field( wp_unslash( $_GET['symbol'] ) ) : '';
    $date_from  = isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '';
    $date_to    = isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '';

    $per_page = 50;
    $page     = isset( $_GET['paged'] ) ? max( 1, intval( $_GET['paged'] ) ) : 1;
    $offset   = ( $page - 1 ) * $per_page;

    $where_clauses = array( '1=1' );
    $prepare_args  = array();

    if ( $account_id ) {
        $where_clauses[] = 'account_id = %d';
        $prepare_args[]  = $account_id;
    }
    if ( $symbol !== '' ) {
        $where_clauses[] = 'symbol = %s';
        $prepare_args[]  = $symbol;
    }
    if ( $date_from !== '' ) {
        $where_clauses[] = 'close_time >= %s';
        $prepare_args[]  = $date_from . ' 00:00:00';
    }
    if ( $date_to !== '' ) {
        $where_clauses[] = 'close_time <= %s';
        $prepare_args[]  = $date_to . ' 23:59:59';
    }

    $where_sql = 'WHERE ' . implode( ' AND ', $where_clauses );

    $count_sql = "SELECT COUNT(*) FROM $table $where_sql";

    if ( $prepare_args ) {
        $total = (int) $wpdb->get_var( $wpdb->prepare( $count_sql, $prepare_args ) );
    } else {
        $total = (int) $wpdb->get_var( $count_sql );
    }

    $total_pages = ( $total > 0 ) ? ceil( $total / $per_page ) : 1;

    $select_sql  = "SELECT * FROM $table $where_sql ORDER BY close_time DESC LIMIT %d OFFSET %d";
    $select_args = $prepare_args;
    $select_args[] = $per_page;
    $select_args[] = $offset;

    if ( $prepare_args ) {
        $trades = $wpdb->get_results( $wpdb->prepare( $select_sql, $select_args ) );
    } else {
        $trades = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table WHERE 1=1 ORDER BY close_time DESC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            )
        );
    }

    ?>
    <div class="wrap">
        <h1>MT4トレード履歴</h1>

        <form method="get" class="mt4mon-filters">
            <input type="hidden" name="page" value="mt4mon-trades" />
            <table class="form-table">
                <tr>
                    <th><label for="mt4mon_account_id">口座番号</label></th>
                    <td>
                        <input id="mt4mon_account_id" type="number" name="account_id"
                               value="<?php echo esc_attr( $account_id ? $account_id : '' ); ?>" />
                    </td>
                </tr>
                <tr>
                    <th><label for="mt4mon_symbol">シンボル</label></th>
                    <td>
                        <input id="mt4mon_symbol" type="text" name="symbol"
                               value="<?php echo esc_attr( $symbol ); ?>" />
                    </td>
                </tr>
                <tr>
                    <th><label for="mt4mon_date_from">期間</label></th>
                    <td>
                        <input id="mt4mon_date_from" type="date" name="date_from"
                               value="<?php echo esc_attr( $date_from ); ?>" />
                        〜
                        <input id="mt4mon_date_to" type="date" name="date_to"
                               value="<?php echo esc_attr( $date_to ); ?>" />
                    </td>
                </tr>
            </table>
            <?php submit_button( 'フィルタ適用' ); ?>
        </form>

        <p>全 <?php echo esc_html( $total ); ?> 件のうち、<?php echo esc_html( $page ); ?> / <?php echo esc_html( $total_pages ); ?> ページ目</p>

        <table class="widefat striped">
            <thead>
                <tr>
                    <th>口座</th>
                    <th>Ticket</th>
                    <th>Symbol</th>
                    <th>Type</th>
                    <th>Lots</th>
                    <th>Open Time</th>
                    <th>Close Time</th>
                    <th>Open</th>
                    <th>Close</th>
                    <th>Profit</th>
                    <th>Magic</th>
                    <th>Comment</th>
                </tr>
            </thead>
            <tbody>
            <?php if ( $trades ) : ?>
                <?php foreach ( $trades as $t ) : ?>
                    <tr>
                        <td><?php echo esc_html( $t->account_id ); ?></td>
                        <td><?php echo esc_html( $t->ticket ); ?></td>
                        <td><?php echo esc_html( $t->symbol ); ?></td>
                        <td><?php echo esc_html( $t->type ); ?></td>
                        <td><?php echo esc_html( $t->lots ); ?></td>
                        <td><?php echo esc_html( $t->open_time ); ?></td>
                        <td><?php echo esc_html( $t->close_time ); ?></td>
                        <td><?php echo esc_html( $t->open_price ); ?></td>
                        <td><?php echo esc_html( $t->close_price ); ?></td>
                        <td><?php echo esc_html( number_format( $t->profit, 2 ) ); ?></td>
                        <td><?php echo esc_html( $t->magic ); ?></td>
                        <td><?php echo esc_html( $t->comment ); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php else : ?>
                <tr><td colspan="12">条件に合うトレードがありません。</td></tr>
            <?php endif; ?>
            </tbody>
        </table>

        <?php if ( $total_pages > 1 ) : ?>
            <div class="tablenav">
                <div class="tablenav-pages">
                    <?php
                    $base_url = remove_query_arg( 'paged' );
                    for ( $p = 1; $p <= $total_pages; $p++ ) :
                        $link = esc_url( add_query_arg( 'paged', $p, $base_url ) );
                        if ( $p === $page ) {
                            echo '<span class="tablenav-pages-navspan">' . esc_html( $p ) . '</span> ';
                        } else {
                            echo '<a class="page-numbers" href="' . $link . '">' . esc_html( $p ) . '</a> ';
                        }
                    endfor;
                    ?>
                </div>
            </div>
        <?php endif; ?>

    </div>
    <?php
}

/**
 * ショートコード: [mt4mon_account account="123456"]
 * 月別集計付きの成績ページ
 */
add_shortcode( 'mt4mon_account', 'mt4mon_shortcode_account' );

function mt4mon_shortcode_account( $atts ) {
    global $wpdb;

    $atts = shortcode_atts(
        array(
            'account' => '',
        ),
        $atts,
        'mt4mon_account'
    );

    $account_id = intval( $atts['account'] );
    if ( ! $account_id ) {
        return '<p>有効な <code>account</code> パラメータを指定してください。</p>';
    }

    $trades_table   = mt4mon_table_trades();
    $accounts_table = mt4mon_table_accounts();

    $account = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT * FROM $accounts_table WHERE account_id = %d ORDER BY id DESC LIMIT 1",
            $account_id
        )
    );

    $trades = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM $trades_table WHERE account_id = %d ORDER BY close_time ASC LIMIT 1000",
            $account_id
        )
    );

    if ( ! $trades ) {
        return '<p>この口座のトレード履歴がまだありません。</p>';
    }

    $total_trades = count( $trades );
    $total_profit = 0.0;
    $win_count    = 0;
    $loss_count   = 0;
    $sum_win      = 0.0;
    $sum_loss     = 0.0;

    $equity_points = array();
    $cum_profit    = 0.0;

    // 月別集計用
    $monthly = array();

    foreach ( $trades as $t ) {
        $p = (float) $t->profit;
        $total_profit += $p;

        $cum_profit += $p;
        $equity_points[] = array(
            'time'   => $t->close_time,
            'equity' => $cum_profit,
        );

        if ( $p > 0 ) {
            $win_count++;
            $sum_win += $p;
        } elseif ( $p < 0 ) {
            $loss_count++;
            $sum_loss += $p;
        }

        // YYYY-MM 形式のキー
        $ym = substr( $t->close_time, 0, 7 );
        if ( ! isset( $monthly[ $ym ] ) ) {
            $monthly[ $ym ] = array(
                'profit' => 0.0,
                'trades' => 0,
                'wins'   => 0,
                'losses' => 0,
            );
        }
        $monthly[ $ym ]['profit'] += $p;
        $monthly[ $ym ]['trades']++;

        if ( $p > 0 ) {
            $monthly[ $ym ]['wins']++;
        } elseif ( $p < 0 ) {
            $monthly[ $ym ]['losses']++;
        }
    }

    $win_rate = ( $total_trades > 0 ) ? ( $win_count / $total_trades * 100.0 ) : 0.0;
    $pf       = ( $sum_loss != 0.0 ) ? ( $sum_win / abs( $sum_loss ) ) : 0.0;

    $labels = array();
    $values = array();
    foreach ( $equity_points as $pt ) {
        $labels[] = $pt['time'];
        $values[] = $pt['equity'];
    }

    $labels_json = wp_json_encode( $labels );
    $values_json = wp_json_encode( $values );

    $latest_trades = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM $trades_table WHERE account_id = %d ORDER BY close_time DESC LIMIT 50",
            $account_id
        )
    );

    // 月別配列を新しい順にソート
    krsort( $monthly );

    ob_start();
    ?>
    <div class="mt4mon-account-summary">
        <h2>口座 <?php echo esc_html( $account_id ); ?> のトレード成績</h2>

        <?php if ( $account ) : ?>
            <p>
                サーバー: <?php echo esc_html( $account->server ); ?><br />
                ステータス: <?php echo esc_html( strtoupper( $account->status ) ); ?><br />
                最終Ping(GMT): <?php echo esc_html( $account->last_ping ); ?>
            </p>
        <?php endif; ?>

        <h3>サマリ</h3>
        <ul>
            <li>トレード回数: <?php echo esc_html( $total_trades ); ?></li>
            <li>累計損益: <?php echo esc_html( number_format( $total_profit, 2 ) ); ?></li>
            <li>勝ちトレード数: <?php echo esc_html( $win_count ); ?></li>
            <li>負けトレード数: <?php echo esc_html( $loss_count ); ?></li>
            <li>勝率: <?php echo esc_html( number_format( $win_rate, 1 ) ); ?> %</li>
            <li>プロフィットファクター: <?php echo esc_html( $pf > 0 ? number_format( $pf, 2 ) : 'N/A' ); ?></li>
        </ul>

        <h3>月別集計</h3>
        <table class="mt4mon-monthly-table">
            <thead>
                <tr>
                    <th>年月</th>
                    <th>トレード数</th>
                    <th>勝率</th>
                    <th>月次損益</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ( $monthly as $ym => $m ) : 
                $m_trades = $m['trades'];
                $m_win    = $m['wins'];
                $m_loss   = $m['losses'];
                $m_winrate = ( $m_trades > 0 ) ? ( $m_win / $m_trades * 100.0 ) : 0.0;
            ?>
                <tr>
                    <td><?php echo esc_html( $ym ); ?></td>
                    <td><?php echo esc_html( $m_trades ); ?></td>
                    <td><?php echo esc_html( number_format( $m_winrate, 1 ) ); ?> %</td>
                    <td><?php echo esc_html( number_format( $m['profit'], 2 ) ); ?></td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>

        <h3>累計損益カーブ（擬似エクイティ）</h3>
        <canvas id="mt4mon-equity-<?php echo esc_attr( $account_id ); ?>" height="140"></canvas>

        <h3>最近のトレード（最新50件）</h3>
        <table class="mt4mon-trades-table">
            <thead>
                <tr>
                    <th>Close Time</th>
                    <th>Symbol</th>
                    <th>Type</th>
                    <th>Lots</th>
                    <th>Open</th>
                    <th>Close</th>
                    <th>Profit</th>
                    <th>Magic</th>
                    <th>Comment</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ( $latest_trades as $t ) : ?>
                <tr>
                    <td><?php echo esc_html( $t->close_time ); ?></td>
                    <td><?php echo esc_html( $t->symbol ); ?></td>
                    <td><?php echo esc_html( $t->type ); ?></td>
                    <td><?php echo esc_html( $t->lots ); ?></td>
                    <td><?php echo esc_html( $t->open_time ); ?></td>
                    <td><?php echo esc_html( $t->close_time ); ?></td>
                    <td><?php echo esc_html( number_format( $t->profit, 2 ) ); ?></td>
                    <td><?php echo esc_html( $t->magic ); ?></td>
                    <td><?php echo esc_html( $t->comment ); ?></td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
    (function() {
        var ctx = document.getElementById('mt4mon-equity-<?php echo esc_js( $account_id ); ?>');
        if (!ctx) return;

        var labels = <?php echo $labels_json; ?>;
        var data   = <?php echo $values_json; ?>;

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: '累計損益',
                    data: data,
                    fill: false,
                    tension: 0.1
                }]
            },
            options: {
                scales: {
                    x: { display: true, title: { display: true, text: 'Close Time' } },
                    y: { display: true, title: { display: true, text: 'Equity (累計損益)' } }
                }
            }
        });
    })();
    </script>
    <?php
    return ob_get_clean();
}
